#!/usr/bin/env python

import es, cmdlib, gamethread, effectlib, cfglib, popuplib, random
from path import path

info = es.AddonInfo()
info.name = 'Player Trails'
info.author = 'craziesT'
info.basename = 'playertrails'
info.description = 'Allows certain players to have trails'

__version__ = '1.0'
addon_path = es.getAddonPath("playertrails") + '/'
public_variable = "%s_version" % info.basename
es.ServerVar(public_variable, __version__, 'Player Trails Version').makepublic()
cfg = cfglib.AddonCFG(addon_path + "/config/config.cfg")

cfg.text("**********************************************************")
cfg.text("**** Player Trails ********* Developed by: .craziesT  ****")
cfg.text("**********************************************************")
cfg.text(" ")
cfg.text("Main Configuration File")
cfg.text(" ")
cfg.text("Edit the settings below to fit the script to your needs,")
cfg.text("if you would like to request a new feature please post in")
cfg.text("in the scripts forum thread or for any support also head to the")
cfg.text("forums. You can also reach me at my steam account: craziest994")
cfg.text(" ")
cfg.text("Load the plugin with: es_load playertrails")
cfg.text(" ")
load_default_colors = cfg.cvar('playertrails_load_default_colors', 0, 'Set this to 1 if you have problems (or an error occures) with the colors not showing up [Happens on Linux]')
admins_only = cfg.cvar('playertrails_admin_only', 1, 'Should only admins have access to this plugin or all players? (1 = Admins, 0 = All)')
delay_frequency = cfg.cvar('playertrails_delay_frequency', 0.02, 'How often should player locations be checked. Lower means it will eat more performance, higher will result in spikier trails.')
trail_command = cfg.cvar('playertrails_trail_command', '!trail', 'The command to access the menu with trails for specific players')
trail_model = cfg.cvar('playertrails_trail_model', 'materials/sprites/laser.vmt', 'The trail model that will be seen ingame')
trail_halo = cfg.cvar('playertrails_trail_model', 'materials/sprites/halo01.vmt', 'The trail halo that will be rendered')
trail_lenght = cfg.cvar('playertrails_trail_time', 3, 'How long should the trails stay before being removed')
trail_width = cfg.cvar('playertrails_trail_width', 20, 'How thick should the trails be on the start')
trail_endwidth = cfg.cvar('playertrails_trail_endwidth', 20, 'How thick should the trails be on the end point')
trail_brightness = cfg.cvar('playertrails_trail_brightness', 255, 'How bright should the trails be? (0-255)')

cfg.write()
colors = {}
default = {
"Red" : (255,0,0),
"Green" : (0,255,0),
"Blue" :(0,0,255),
"Purple" :(128,0,128),
"Yellow" :(255,255,0),
"Orange" :(255,165,0),
"White" :(255,255,255),
"Grey" :(50,50,50),
"Silver" :(192,192,192),
"BlueViolet" :(138,43,226),
"HunterGreen" :(142,35,35),
"SteelBlue" :(35,107,142),
"Antique White" :(205,192,176),
"Pink" :(255,192,203),
"Scarlet" :(140,23,23),
"SealGreen" :(128,255,165),
"OliveBrown" :(205,176,43),
"SeaBlue" :(43,176,205),
"PaleTurquoise" :(175,238,238),
"FireBrick" :(178,34,34),
"AliceBlue" :(240,248,255),
"CornSilk" :(255,248,220),
"Cyan" :(0,255,255),
"Sienna" :(180,82,45),
"SandyBrown" :(244,164,96),
"Plum" :(221,160,221),
"OliveDrab" :(107,142,35),
"Orchird" :(218,112,214),
"NavajoWhite" :(255,222,173),
"LawnGreen" :(124,252,0),
"DimGrey" :(105,105,105),
"DarkKhaki" :(189,183,107),
"Turquoise" :(64,224,208),   
}
trails = {}
admins = []
prefix = '#lightgreenPlayer Trails #default: #green'

#############################################

class PlayerTrail(object):
    def __init__(self, userid):
        self.userid = userid
        self.status = False
        self.stop = False
        self.random = False
        self.random2 = False
        self.random3 = False
        self.location = (0, 0, 0)
    
    def showMenu(self):
        self.popup = popuplib.easymenu('trailp_%s' % self.userid, None, self.select)
        self.popup.settitle('Player Trails')
        self.popup.setdescription(':: Select a color')
        self.popup.addoption('cancel', 'Stop displaying trail')
        self.popup.addoption('random', 'Random Colors')
        self.popup.addoption('random2', 'Quick Random Switch')
        self.popup.addoption('random3', 'Slow Random Switch')
        for color in colors:
            self.popup.addoption(colors[color], color)
            
        
        self.popup.send(self.userid)
        
    def select(self, userid, choice, popupid):
        if choice != 'cancel' and choice != 'random' and choice != 'random2' and choice != 'random3':
            self.random = False
            self.random2 = False
            if self.random3:
                gamethread.cancelDelayed('trail_random_%s' % self.userid)
            self.random3 = False
            self.r, self.g, self.b = choice
            if self.status == False:
                self.status = True
                self.start()
                es.tell(self.userid, '#multi', prefix + 'You have just started your trail.')
        elif choice == 'random':
            if self.status == False:
                self.status = True
            self.random = True
            self.random2 = False
            if self.random3:
                gamethread.cancelDelayed('trail_random_%s' % self.userid)
            self.random3 = False
            if self.random:
                if load_default_colors:
                    color = random.choice(default.keys())
                    self.r, self.g, self.b = default[color]
                else:
                    color = random.choice(colors.keys())
                    self.r, self.g, self.b = colors[color]
            self.start()
            es.tell(self.userid, '#multi', prefix + 'You have just started a random trail color.')
        elif choice == 'random2':
            if self.status == False:
                self.status = True
            self.random = False
            self.random2 = True
            if self.random3:
                gamethread.cancelDelayed('trail_random_%s' % self.userid)
            self.random3 = False
            self.start()
            es.tell(self.userid, '#multi', prefix + 'You have just started the quick random switch option.')
        elif choice == 'random3':
            if self.status == False:
                self.status = True
            self.random = False
            self.random2 = False
            self.random3 = True
            self.doRandomLoop(True)
            self.start()
            es.tell(self.userid, '#multi', prefix + 'You have just started the slow random switch option.')
        else:
            self.random = False
            self.random2 = False
            self.random3 = False
            if self.status != False:
                self.status = False
                es.tell(self.userid, '#multi', prefix + 'You have stopped displaying your trail.')
            else:
                es.tell(self.userid, '#multi', prefix + 'You already don\'t have a trail.')
                
        
    def doRandomLoop(self, loop=None):
        if not loop:
            if self.random:
                if load_default_colors:
                    color = random.choice(default.keys())
                    self.r, self.g, self.b = default[color]
                else:
                    color = random.choice(colors.keys())
                    self.r, self.g, self.b = colors[color]
        else:
            if self.random3:
                if load_default_colors:
                    color = random.choice(default.keys())
                    self.r, self.g, self.b = default[color]
                else:
                    color = random.choice(colors.keys())
                    self.r, self.g, self.b = colors[color]
            gamethread.delayedname(3, 'trail_random3_%s' % self.userid, self.doRandomLoop, True)
        
    def start(self):
        self.location = es.getplayerlocation(self.userid)
        gamethread.delayedname(0.02, 'trail_%s' % self.userid, self.check)
        
    def check(self):
        if not self.stop and self.status:
            new_location = es.getplayerlocation(self.userid)
            if new_location != self.location:
                if self.random2:
                    if load_default_colors:
                        color = random.choice(default.keys())
                        self.r, self.g, self.b = default[color]
                    else:
                        color = random.choice(colors.keys())
                        self.r, self.g, self.b = colors[color]
                self.draw(self.location, new_location)
        
            self.location = new_location
        gamethread.delayedname(delay_frequency, 'trail_%s' % self.userid, self.check)
        
    def playerDeath(self):
        self.stop = True
        
    def playerSpawn(self):
        self.location = es.getplayerlocation(self.userid)
        self.stop = False
        self.check()
        
    def roundEnd(self):
        self.stop = True
        
    def cancel(self):
        gamethread.cancelDelayed('trail_%s' % self.userid)
        if self.random3:
            gamethread.cancelDelayed('trail_random3_%s' % self.userid)
        self.status = False
        
    def draw(self, coord1, coord2):
        effectlib.drawLine(coord1, coord2, trail_model, trail_halo, trail_lenght,
            trail_width, trail_endwidth, self.r, self.g, self.b, trail_brightness, speed=5, fadelength=100, noise=0, framestart=0, framerate=0)

#############################################

def registerTrail(userid):
    if userid not in trails:
        trails[userid] = PlayerTrail(userid)
        
    return trails[userid]

#############################################

def load():
    # admin support
    cmdlib.registerSayCommand('!trail', trail, 'Player Trails Command')
    importAdmins()
    importColors()
    cfg.execute()
    
def unload():
    cmdlib.unregisterSayCommand('!trail')
    for userid in es.getUseridList():
        if userid in trails:
            pTrail = registerTrail(userid)
            pTrail.cancel()
    
def trail(userid, args):
    if es.getplayersteamid(userid) in admins or not admins_only:
        pTrail = registerTrail(userid)
        pTrail.showMenu()
    else:
        es.tell(userid, '#multi', prefix + 'You have no permission to use this command.')
    
def player_death(ev):
    userid = int(ev['userid'])
    if userid in trails:
        pTrail = registerTrail(userid)
        pTrail.playerDeath()
    
def player_spawn(ev):
    userid = int(ev['userid'])
    if userid in trails:
        pTrail = registerTrail(userid)
        pTrail.playerSpawn()
        
def player_diconnect(ev):
    userid = int(ev['userid'])
    if userid in trails:
        pTrail = registerTrail(userid)
        pTrail.cancel()
    
def round_end(ev):
    for userid in es.getUseridList():
        if userid in trails:
            pTrail = registerTrail(userid)
            pTrail.roundEnd()
            
def es_map_start(ev):
    for userid in es.getUseridList():
        if userid in trails:
            pTrail = registerTrail(userid)
            pTrail.cancel()

############################################

def importAdmins():
    adminFile = addon_path + 'config/admins.txt'
    if path(adminFile).isfile():
        file = open(adminFile, 'r')
        for line in filter(lambda line: not line.startswith('//') and not line.startswith('\n'), file.readlines()):
            admins.append(line.replace('\n', ''))
        es.dbgmsg(0, '[Player Trails] Admins have been succesfully added!')
    else:
        es.dbgmsg(0, '[Player Trails] There is no admin file, creating default file...!')
        file = open(adminFile, 'w')
        file.write('// Input your admins here, they will have access to !trails\nSTEAM_0:1:11000\nSTEAM_0:1:15131822')
        file.close()
        importAdmins()
        
def importColors():
    global colors
    if load_default_colors:
        colors = default
        es.dbgmsg(0, '[Player Trails] Default Colors added, because you have chosen this setting!')
        return
    colorsFile = addon_path + 'config/colors.txt'
    if path(colorsFile).isfile():
        file = open(colorsFile, 'r')
        for line in filter(lambda line: not line.startswith('//') and not line.startswith('\n'), file.readlines()):
            name_sep = line.find('(')
            name = line[:name_sep].strip().replace('(','')
            if name in colors: continue
            rgb = line[name_sep:].strip().replace(')', '').replace('(', '')
            r, g, b = rgb.split(',')
            colors[name] = (r.strip(), g.strip(), b.strip())
        es.dbgmsg(0, '[Player Trails] Colors have been succesfully added!')
    else:
        es.dbgmsg(0, '[Player Trails] There is no colors file, creating default file...!')
        file = open(colorsFile, 'w')
        file.write('// Input your colors here, players will have access to them via !trails\n// Red\n255 0 0\n// Green\n0 255 0\n// Blue\n0 0 255')
        file.close()
        importColors()
    