/**
 * ==========================================================================
 * SourceMod RockTheMode CS:GO
 *
 * by Sheepdude
 *
 * SourceMod Forums Plugin Thread URL:
 * https://forums.alliedmods.net/showthread.php?t=196148
 *
 * This plugin is specifically designed to work in-sync with the NextMapMode
 * plugin here: https://forums.alliedmods.net/showthread.php?t=193912
 * These two plugins, when combined, allow server administrators to set a
 * map rotation which includes predefined game modes, with the option for
 * players to switch modes for the current map, if they wish.
 *
 * I used the SourceMod RockTheVote plugin as the basis for this plugin.
 *
 * This plugin allows your players to switch the game mode by popular vote.
 * Similar to the RockTheVote plugin, players can opt to RockTheMode by
 * typing "rtm" or "rockthemode" in-game. When a large enough percentage of
 * players have opted to vote, a vote menu appears, allowing the players to
 * select between Casual, Competitive, Arms Race, and Demolition. When the
 * winner has been decided, the map will restart with the chosen setting.
 *
 * You can set which game modes appear in the vote by modifying rtm.cfg.
 *
 * Admins can change the mode immediately with sm_rtm_mode <mode>, or force
 * the vote to appear with sm_rtm_forcevote.
 * Example usage: sm_rtm_mode Demolition
 *
 * CHANGELOG
 *
 * Version 0.8.0 (16 September 2012)
 * -Initial Version
 *
 * Version 0.8.5 (16 September 2012)
 * -Redesigned to function correctly with NextMapMode. NMM is disabled when
 *  an RTM vote is successful, so that the changes aren't overwritten.
 *
 * Version 0.9.0 (17 September 2012)
 * -Added support for manually selecting which game modes appear on the vote.
 *  Admins can change this in /csgo/cfg/sourcemod/rtm.cfg.
 *
 * Version 1.0.0 (18 September 2012)
 * -Map no longer changes if the chosen mode is the currently active mode.
 * -Admins with map changing privileges can now use sm_rtm_mode to change the
 *  mode immediately. Acceptable arguments are casual, competitive, armsrace,
 *  arms_race, and demolition.
 * -Winner voting percentage is now displayed after a successful vote.
 * -Added cvar sm_rtm_initialwindow to disable rtm voting after a certain
 *  amount of time has elapsed. This prevents players from using rtm to
 *  restart the map late in the game.
 * 
 * Version 1.1.0 (29 September 2012)
 * -Previously, "sm_rtm_mode arms_race" would not switch modes correctly if
 *  any part of "arms_race" was capitalized. This has been fixed.
 * -Changed to work correctly with latest NextMapMode build.
 * -Convar sm_rtm_postvoteaction changed to sm_rtm_enable to more accurately
 *  reflect its purpose.
 * -Code cleaned up and optimized.
 * 
 * Version 1.1.1 (3 October 2012)
 * -Made plugin version public (sm_rtm_version)
 *
 * Version 1.1.2 (14 October 2012)
 * Current Version
 * -Previously, rockthemode was reset and delayed if another sourcemod vote
 *  was in progress. This has been fixed.
 * -Added admin command "sm_rtm_forcevote" - Forces a rockthemode vote.
 *
 */

#include <sourcemod>
#pragma semicolon 1

new Handle:h_CVAR_ENABLE = INVALID_HANDLE;
new Handle:h_CVAR_NEEDED = INVALID_HANDLE;
new Handle:h_CVAR_MINPLAYERS = INVALID_HANDLE;
new Handle:h_CVAR_INITIALDELAY = INVALID_HANDLE;
new Handle:h_CVAR_INITIALWINDOW = INVALID_HANDLE;
new Handle:h_CVAR_INTERVAL = INVALID_HANDLE;
new Handle:h_ALLOW_CASUAL = INVALID_HANDLE;
new Handle:h_ALLOW_COMPETITIVE = INVALID_HANDLE;
new Handle:h_ALLOW_ARMSRACE = INVALID_HANDLE;
new Handle:h_ALLOW_DEMOLITION = INVALID_HANDLE;
new Handle:h_SERVER_GAMETYPE= INVALID_HANDLE;
new Handle:h_SERVER_GAMEMODE = INVALID_HANDLE;
new Handle:h_NEXTMAPMODE_ENABLE = INVALID_HANDLE;		// Used for compatibility with NMM plugin.

new String:g_PLUGIN_VERSION[16] = "1.1.2";
new String:g_NEXTMODE[32] = "";
new bool:g_CANRTM = false;			// True if RTM loaded maps and is active.
new bool:g_RTMALLOWED = false;		// True if RTM is available to players. Used to delay rtm votes.
new bool:g_WINDOWPASSED = false;	// True if RTM voting window has passed. Used to disable RTM.

new g_VOTERS = 0;				// Total voters connected. Doesn't include fake clients.
new g_VOTES = 0;				// Total number of "say rtm" votes
new g_VOTESNEEDED = 0;			// Necessary votes before mode vote begins. (voters * percent_needed)
new bool:g_VOTED[MAXPLAYERS+1] = {false, ...};
new bool:g_INCHANGE = false;
new g_SWITCHEDNMM = 0;			// If RTM vote succeeded last map and NMM needs to be reenabled.
new g_GAMETYPE = 0;				// 0 - Classic, 1 - Arsenal
new g_GAMEMODE = 0;				// 0 - Casual or Arms Race, 1 - Competitive or Demolition (dependent on game_type)

public Plugin:myinfo =
{
	name = "Rock The Mode",
	author = "Sheepdude",
	description = "Provides RTM Game Mode Voting for CS:GO",
	version = g_PLUGIN_VERSION,
	url = "http://www.clan-psycho.com"
};

public OnPluginStart()
{
	// Load Translation files
	LoadTranslations("rockthemode.phrases");
	LoadTranslations("basevotes.phrases");

	// Create cvars for the plugin
	CreateConVar("sm_rtm_version", g_PLUGIN_VERSION, "Provides RTM Game Mode Voting for CS:GO", FCVAR_DONTRECORD|FCVAR_NOTIFY|FCVAR_PLUGIN|FCVAR_REPLICATED|FCVAR_SPONLY);
	h_CVAR_ENABLE = CreateConVar("sm_rtm_enable", "1", "Enable or disable plugin, 1 - enable, 0 - disable", FCVAR_NOTIFY, true, 0.0, true, 1.0);
	h_CVAR_NEEDED = CreateConVar("sm_rtm_needed", "0.60", "Percentage of players needed to RockTheMode (Def 60%)", 0, true, 0.05, true, 1.0);
	h_CVAR_MINPLAYERS = CreateConVar("sm_rtm_minplayers", "0", "Number of players required before RTM will be enabled", 0, true, 0.0, true, float(MAXPLAYERS));
	h_CVAR_INITIALDELAY = CreateConVar("sm_rtm_initialdelay", "20.0", "Time (in seconds) before first RTM can be held", 0, true, 0.00);
	h_CVAR_INITIALWINDOW = CreateConVar("sm_rtm_initialwindow", "240.0", "Time (in seconds) after map start to disable RTM vote", 0, true, 0.00);
	h_CVAR_INTERVAL = CreateConVar("sm_rtm_interval", "240.0", "Time (in seconds) after a failed RTM before another can be held", 0, true, 0.00);
	h_ALLOW_CASUAL = CreateConVar("sm_rtm_allow_casual", "1", "Does Casual appear in the vote (0-no, 1-yes)", 0, true, 0.00, true, 1.0);
	h_ALLOW_COMPETITIVE = CreateConVar("sm_rtm_allow_competitive", "1", "Does Competitive appear in the vote (0-no, 1-yes)", 0, true, 0.00, true, 1.0);
	h_ALLOW_ARMSRACE = CreateConVar("sm_rtm_allow_armsrace", "1", "Does Arms Race appear in the vote (0-no, 1-yes)", 0, true, 0.00, true, 1.0);
	h_ALLOW_DEMOLITION = CreateConVar("sm_rtm_allow_demolition", "1", "Does Demolition appear in the vote (0-no, 1-yes)", 0, true, 0.00, true, 1.0);

	// Find cvars so that we can change them
	h_NEXTMAPMODE_ENABLE = FindConVar("sm_nmm_enable");
	h_SERVER_GAMETYPE = FindConVar("game_type");
	h_SERVER_GAMEMODE = FindConVar("game_mode");

	// Register commands to let players Rockthemode
	RegConsoleCmd("say", command_Say);
	RegConsoleCmd("say_team", command_Say);
	RegConsoleCmd("sm_rtm", command_RTM);

	// Admin commands
	RegAdminCmd("sm_rtm_mode", command_ChangeMode, ADMFLAG_CHANGEMAP, "Immediately changes the current game mode.");
	RegAdminCmd("sm_rtm_forcevote", command_ForceVote, ADMFLAG_CHANGEMAP, "Forces a Rockthemode vote.");

	// Register console command to display plugin version
	RegConsoleCmd("rtm_version", command_Version);
	AutoExecConfig(true, "rtm");
}

public OnMapStart()
{
	// If initial window is set, start a timer that disables rtm after the window has expired
	new Float:tempWindow = GetConVarFloat(h_CVAR_INITIALWINDOW);
	if(tempWindow != 0)
		CreateTimer(tempWindow, timer_WindowRTM, _, TIMER_FLAG_NO_MAPCHANGE);
	// Instantiate voter information
	g_VOTERS = 0;
	g_VOTES = 0;
	g_VOTESNEEDED = 0;
	g_INCHANGE = false;
	// For compatibility with NextMapMode plugin - reenables NMM if it was disabled by RTM last map
	if(h_NEXTMAPMODE_ENABLE != INVALID_HANDLE && g_SWITCHEDNMM == 1)
		SetConVarInt(h_NEXTMAPMODE_ENABLE, 1);
	// Handle late load
	for (new i = 1; i <= MaxClients; i++)
		if (IsClientConnected(i))
			OnClientConnected(i);	
}

public OnMapEnd()
{
	// Reset data when the map ends
	g_CANRTM = false;	
	g_RTMALLOWED = false;
	g_WINDOWPASSED = false;
}

public OnConfigsExecuted()
{	
	g_CANRTM = true;
	// Disable rockthemode until initial delay has passed
	g_RTMALLOWED = false;
	CreateTimer(GetConVarFloat(h_CVAR_INITIALDELAY), timer_DelayRTM, _, TIMER_FLAG_NO_MAPCHANGE);
}

// When client connects, increase max voters and votes needed
public OnClientConnected(client)
{
	if(IsFakeClient(client))
		return;
	g_VOTED[client] = false;
	g_VOTERS++;
	g_VOTESNEEDED = RoundToFloor(float(g_VOTERS) * GetConVarFloat(h_CVAR_NEEDED));
	return;
}

// When client disconnects, decrease max voters and votes needed
public OnClientDisconnect(client)
{
	if(IsFakeClient(client))
		return;
	if(g_VOTED[client])
		g_VOTES--;
	g_VOTERS--;
	g_VOTESNEEDED = RoundToFloor(float(g_VOTERS) * GetConVarFloat(h_CVAR_NEEDED));
	if (!g_CANRTM)
		return;
	// Starts vote if a client disconnect cause vote count to exceed the needed ratio
	if (g_VOTES && 
		g_VOTERS && 
		g_VOTES >= g_VOTESNEEDED && 
		g_RTMALLOWED ) 
	{
		if (GetConVarInt(h_CVAR_ENABLE) == 0)
			return;
		startRTM();
	}	
}

// Handler for "sm_rtm" console command
public Action:command_RTM(client, args)
{
	if (!g_CANRTM || !client)
		return Plugin_Handled;
	attemptRTM(client);
	return Plugin_Handled;
}

// Handler for "say" command - checks for "rtm" or "rockthemode" in chat string
public Action:command_Say(client, args)
{
	if (!g_CANRTM || !client)
		return Plugin_Continue;
	decl String:text[192];
	if (!GetCmdArgString(text, sizeof(text)))
		return Plugin_Continue;
	new startidx = 0;
	if(text[strlen(text)-1] == '"')
	{
		text[strlen(text)-1] = '\0';
		startidx = 1;
	}
	new ReplySource:old = SetCmdReplySource(SM_REPLY_TO_CHAT);
	// If client chat string matches "rtm" or "rockthemode", try to register their vote
	if (strcmp(text[startidx], "rtm", false) == 0 || strcmp(text[startidx], "rockthemode", false) == 0)
		attemptRTM(client);
	SetCmdReplySource(old);
	return Plugin_Continue;	
}

// Handler for "sm_rtm_mode" admin command
public Action:command_ChangeMode(client, args)
{
	if(args < 1)
	{
		PrintToConsole(client, "[SM] Usage: sm_rtm_mode <mode>");
		return Plugin_Handled;
	}
	if(IsVoteInProgress())
		return Plugin_Handled;
	new String:argstring[256];
	GetCmdArgString(argstring, sizeof(argstring));
	setMode(argstring);
	if(strcmp(g_NEXTMODE, "") == 0)
		return Plugin_Handled;
	restartMap();
	return Plugin_Handled;
}

// Handler for "sm_rtm_forcevote" admin command
public Action:command_ForceVote(client, args)
{
	startRTM();
	return Plugin_Handled;
}

// Handler for "rtm_version" console command
public Action:command_Version(client, args)
{
	if(GetCmdReplySource() == SM_REPLY_TO_CHAT)
		PrintToChat(client, "Please view your console for more information.");
	PrintToConsole(client, "RockTheMode Information:\n   Version: %s   Author: Sheepdude", g_PLUGIN_VERSION);
	PrintToConsole(client, "   Website: http://www.clan-psycho.com\n   Compiled Time: 07:47, 16 October 2012");
	return Plugin_Handled;
}

// Attempt to register client vote
attemptRTM(client)
{
	if (!g_RTMALLOWED  || (GetConVarInt(h_CVAR_ENABLE) == 0))
	{
		ReplyToCommand(client, "[SM] %t", "RTM Not Allowed");
		return;
	}	
	if (GetClientCount(true) < GetConVarInt(h_CVAR_MINPLAYERS))
	{
		ReplyToCommand(client, "[SM] %t", "Minimal Players Not Met");
		return;			
	}
	if (g_VOTED[client])
	{
		ReplyToCommand(client, "[SM] %t", "Already Voted", g_VOTES, g_VOTESNEEDED);
		return;
	}	
	new String:name[64];
	GetClientName(client, name, sizeof(name));
	g_VOTES++;
	g_VOTED[client] = true;
	PrintToChatAll("[SM] %t", "RTM Requested", name, g_VOTES, g_VOTESNEEDED);
	// Start rockthemode vote if registering the client vote caused the total votes to exceed the needed ratio
	if (g_VOTES >= g_VOTESNEEDED)
		startRTM();
}

// Starts the mode voting once enough client votes have been received
startRTM()
{
	// Do not start vote if plugin is already changing map or if another sm vote is in progress
	if (g_INCHANGE || IsVoteInProgress())
		return;
	// Create and display rockthemode vote menu
	doVoteMenu();
	// Reset player votes and disallow voting for rockthemode until the delay interval has passed
	g_RTMALLOWED = false;
	resetRTM();
	CreateTimer(GetConVarFloat(h_CVAR_INTERVAL), timer_DelayRTM, _, TIMER_FLAG_NO_MAPCHANGE);
}
 
// Create the mode voting menu
doVoteMenu()
{
	new Handle:menu = CreateMenu(handleVoteMenu);
	SetMenuTitle(menu, "Choose a game mode:");
	if(GetConVarInt(h_ALLOW_CASUAL) == 1)
		AddMenuItem(menu, "Casual", "Casual");
	if(GetConVarInt(h_ALLOW_COMPETITIVE) == 1)
		AddMenuItem(menu, "Competitive", "Competitive");
	if(GetConVarInt(h_ALLOW_ARMSRACE) == 1)
		AddMenuItem(menu, "Arms Race", "Arms Race");
	if(GetConVarInt(h_ALLOW_DEMOLITION) == 1)
		AddMenuItem(menu, "Demolition", "Demolition");
	// Do not vote if vote menu is empty
	if(GetMenuItemCount(menu) == 0)
	{
		CloseHandle(menu);
		resetRTM();
		return;
	}
	SetMenuExitButton(menu, false);
	VoteMenuToAll(menu, 20);
}

//Handler for the rockthemode vote menu
public handleVoteMenu(Handle:menu, MenuAction:action, param1, param2) 
{
	// Close vote menu handle after the vote has stopped
	if (action == MenuAction_End) 
		CloseHandle(menu);
	// Handle the results of the vote
	else if (action == MenuAction_VoteEnd)
	{
		// Get vote winner and set the next map mode accordingly
		decl String:nextMode[16];
		GetMenuItem(menu, param1, nextMode, sizeof(nextMode));
		setMode(nextMode);
		new votes;
		new totalVotes;
		GetMenuVoteInfo(param2, votes, totalVotes);
		new Float:percent = FloatDiv(float(votes),float(totalVotes));
		PrintToChatAll("[SM] %t", "Vote Successful", RoundToNearest(100.0*percent), totalVotes);
		restartMap();
	}
}

// Sets variables for the next mode based on either the vote winner or the admin argstring for sm_rtm_mode
setMode(const String:nextMode[])
{
	g_GAMETYPE = 0;
	g_GAMEMODE = 0;
	if(strcmp(nextMode, "Casual", false) == 0)		
		g_NEXTMODE = "Casual";
	else if(strcmp(nextMode, "Competitive", false) == 0)
	{
		g_GAMEMODE = 1;
		g_NEXTMODE = "Competitive";
	}
	else if(strcmp(nextMode, "Armsrace", false) == 0 || strcmp(nextMode, "arms_race", false) == 0 || strcmp(nextMode, "Arms Race", false) == 0)
	{
		g_GAMETYPE = 1;
		g_NEXTMODE = "Arms Race";
	}
	else if(strcmp(nextMode, "Demolition", false) == 0)
	{
		g_GAMETYPE = 1;
		g_GAMEMODE = 1;
		g_NEXTMODE = "Demolition";
	}
	else
		g_NEXTMODE = "";
}

public restartMap()
{
		// If current game mode won, don't change map and delay rockthemode vote
		if(GetConVarInt(h_SERVER_GAMETYPE) == g_GAMETYPE && GetConVarInt(h_SERVER_GAMEMODE) == g_GAMEMODE)
		{
			PrintToChatAll("[SM] No action required. Game mode is already %s.", g_NEXTMODE);
			CreateTimer(GetConVarFloat(h_CVAR_INTERVAL), timer_DelayRTM, _, TIMER_FLAG_NO_MAPCHANGE);
			resetRTM();
			return;
		}
		// For compability with NextMapMode plugin. If NMM is enabled, disable it.
		if(h_NEXTMAPMODE_ENABLE != INVALID_HANDLE)
		{
			g_SWITCHEDNMM = GetConVarInt(h_NEXTMAPMODE_ENABLE);
			SetConVarInt(h_NEXTMAPMODE_ENABLE, 0);
		}
		// Change game mode and game type, then restart the map
		SetConVarInt(h_SERVER_GAMETYPE, g_GAMETYPE);
		SetConVarInt(h_SERVER_GAMEMODE, g_GAMEMODE);
		PrintToChatAll("[SM] %t", "Changing Modes", g_NEXTMODE);
		CreateTimer(5.0, timer_ChangeMap, _, TIMER_FLAG_NO_MAPCHANGE);
}

// Resets client vote status once a rockthemode vote has been initiated
resetRTM()
{
	g_VOTES = 0;
	for (new i = 1; i <= MAXPLAYERS; i++)
		g_VOTED[i] = false;
}

// Disables rockthemode after the window time has passed
public Action:timer_WindowRTM(Handle:timer)
{
	g_WINDOWPASSED = true;
	g_RTMALLOWED = false;
}

// Enables rockthemode after the initial delay has passed
public Action:timer_DelayRTM(Handle:timer)
{
	if(!g_WINDOWPASSED)
		g_RTMALLOWED = true;
}

// Restarts the map
public Action:timer_ChangeMap(Handle:timer)
{
	g_INCHANGE = false;
	new String:currentMap[32];
	GetCurrentMap(currentMap, sizeof(currentMap));
	ForceChangeLevel(currentMap, "RTM changing mode manually");
	return Plugin_Stop;
}